﻿import { Head, useForm, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card } from '@/components/ui/card';
import { Textarea } from '@/components/ui/textarea';
import { Checkbox } from '@/components/ui/checkbox';
import { FormEvent } from 'react';

interface ContactList {
  id: number;
  name: string;
  description?: string;
}

interface Props {
  lists: ContactList[];
}

export default function ContactCreate({ lists }: Props) {
  const { data, setData, post, processing, errors } = useForm({
    first_name: '',
    last_name: '',
    phone_number: '',
    email: '',
    company: '',
    notes: '',
    list_ids: [] as number[],
  });

  const handleListToggle = (listId: number) => {
    const newListIds = data.list_ids.includes(listId)
      ? data.list_ids.filter(id => id !== listId)
      : [...data.list_ids, listId];
    setData('list_ids', newListIds);
  };

  const handleSubmit = (e: FormEvent) => {
    e.preventDefault();
    post('/contacts');
  };

  return (
    <AppLayout>
      <Head title="Add Contact" />

      <div className="max-w-2xl mx-auto space-y-6">
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-3xl font-bold">Add Contact</h1>
            <p className="text-muted-foreground">Create a new contact</p>
          </div>
          <Button variant="outline" onClick={() => router.get('/contacts')}>
            Cancel
          </Button>
        </div>

        <Card className="p-6">
          <form onSubmit={handleSubmit} className="space-y-6">
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="first_name">First Name *</Label>
                <Input
                  id="first_name"
                  value={data.first_name}
                  onChange={(e) => setData('first_name', e.target.value)}
                  required
                />
                {errors.first_name && <p className="text-sm text-destructive">{errors.first_name}</p>}
              </div>

              <div className="space-y-2">
                <Label htmlFor="last_name">Last Name *</Label>
                <Input
                  id="last_name"
                  value={data.last_name}
                  onChange={(e) => setData('last_name', e.target.value)}
                  required
                />
                {errors.last_name && <p className="text-sm text-destructive">{errors.last_name}</p>}
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="phone_number">Phone Number *</Label>
              <Input
                id="phone_number"
                type="tel"
                value={data.phone_number}
                onChange={(e) => setData('phone_number', e.target.value)}
                placeholder="+1234567890"
                required
              />
              {errors.phone_number && <p className="text-sm text-destructive">{errors.phone_number}</p>}
            </div>

            <div className="space-y-2">
              <Label htmlFor="email">Email</Label>
              <Input
                id="email"
                type="email"
                value={data.email}
                onChange={(e) => setData('email', e.target.value)}
              />
              {errors.email && <p className="text-sm text-destructive">{errors.email}</p>}
            </div>

            <div className="space-y-2">
              <Label htmlFor="company">Company</Label>
              <Input
                id="company"
                value={data.company}
                onChange={(e) => setData('company', e.target.value)}
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="notes">Notes</Label>
              <Textarea
                id="notes"
                value={data.notes}
                onChange={(e) => setData('notes', e.target.value)}
                rows={4}
              />
            </div>

            {/* Contact Lists Selection */}
            {lists && lists.length > 0 && (
              <div className="space-y-2">
                <Label>Add to Contact Lists (Optional)</Label>
                <div className="border rounded-md p-4 space-y-3 max-h-48 overflow-y-auto">
                  {lists.map((list) => (
                    <div key={list.id} className="flex items-center space-x-3">
                      <Checkbox
                        id={`list-${list.id}`}
                        checked={data.list_ids.includes(list.id)}
                        onCheckedChange={() => handleListToggle(list.id)}
                      />
                      <Label htmlFor={`list-${list.id}`} className="cursor-pointer font-normal">
                        <div>
                          <p className="font-medium">{list.name}</p>
                          {list.description && (
                            <p className="text-sm text-muted-foreground">{list.description}</p>
                          )}
                        </div>
                      </Label>
                    </div>
                  ))}
                </div>
                {data.list_ids.length > 0 && (
                  <p className="text-sm text-muted-foreground">
                    Contact will be added to {data.list_ids.length} list{data.list_ids.length > 1 ? 's' : ''}
                  </p>
                )}
              </div>
            )}

            <div className="flex justify-end gap-2">
              <Button type="button" variant="outline" onClick={() => router.get('/contacts')}>
                Cancel
              </Button>
              <Button type="submit" disabled={processing}>
                {processing ? 'Creating...' : 'Create Contact'}
              </Button>
            </div>
          </form>
        </Card>
      </div>
    </AppLayout>
  );
}
